using System;
using System.Text;
using System.Text.RegularExpressions;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class represents a version number in 'NN.NN......NN' format. It's facilitated to 
	/// contain various version numbers used in VistALink messages.
	/// Number of number groups in version number is not limited, however, every number group 
	/// may not contain more than 5 digits. 
	/// </summary>
	public class VersionNumber : IComparable
	{
		private const char VersionGroupSeparator = '.';
		private readonly int[] _versionGroups;

		private static readonly Regex _versionNumberCheckRegex = 
			new Regex( @"\A\d{1,5}(\.\d{1,5})*\Z" );

		/// <summary>
		/// Constructor accepting version number expressed as string. 
		/// </summary>
		/// <param name="versionNumberAsString">
		///		Version number as string in 'NN.NN....NN" format where 
		///		'NN' is a version number group of no more than 5 digits. Version number may contain
		///		arbitrary number of groups. 
		/// </param>
		public VersionNumber( string versionNumberAsString )
		{
			if( versionNumberAsString == null )
				throw( new ArgumentNullException( "versionNumberAsString" ) );

			if( !_versionNumberCheckRegex.IsMatch( versionNumberAsString ) )
				throw( new ArgumentOutOfRangeException( SR.Exceptions.VersionNumberIncorrectInputStringFormat( versionNumberAsString ) ) );

			string [] _versionGroupsStrs = versionNumberAsString.Split( VersionGroupSeparator );

			_versionGroups = (int[])(Array.CreateInstance( typeof(int), _versionGroupsStrs.Length ));

			for( int i = 0; i < _versionGroupsStrs.Length; i++ )
				_versionGroups[i] = Int32.Parse( _versionGroupsStrs[i] );
		}

		/// <summary>
		/// Deserialization factory method parsing supplied version number string and returning a new instance of 
		/// VersionNumber object. It differs from instance constructor only in throwing 
		/// StringParseException instead of ArgumentOutOfRangeException in case of error.
		/// </summary>
		/// <param name="versionNumberString">
		///		Version number as string in 'NN.NN....NN" format where 
		///		'NN' is a version number group of no more than 5 digits. Version number may contain
		///		arbitrary number of groups. 
		/// </param>
		/// <returns>New instance of VersionNumber created from supplied string.</returns>
		public static VersionNumber Parse( string versionNumberString )
		{
			if( versionNumberString == null )
				throw( new ArgumentNullException( "versionNumberString" ) );

			try
			{
				return new VersionNumber( versionNumberString );
			}
			catch( ArgumentOutOfRangeException xcp )
			{
				throw( new StringParseException( xcp.Message ) );
			}
		}

		/// <summary>
		/// Implementation of standard ToString() method converting version number to string. 
		/// </summary>
		/// <returns>String representation of version number in 'NN.NN...NN.NN' format.</returns>
		public override string ToString()
		{
			StringBuilder _sb = new StringBuilder();

			_sb.Append( _versionGroups[0].ToString() );
			
			for( int i = 1; i < _versionGroups.Length; i++ )
			{
				_sb.Append( VersionGroupSeparator );
				_sb.Append( _versionGroups[i].ToString() );
			}

			return _sb.ToString();
		}

		/// <summary>
		/// Implementation of standard CompareTo method from IComparable interface.
		/// </summary>
		/// <param name="obj">Object to compare.</param>
		/// <returns>
		///		Positive number if this VersionNumber is greater that supplied VersionNumber. 
		///		0 if they are equal. Negative number otherwise.
		///	</returns>
		public int CompareTo( object obj )
		{			
			if( obj == null )
				return 0;

			if( !( obj is VersionNumber ) )
				throw( new ArgumentException( SR.Exceptions.VersionNumberObjectToCompareIsOfWrongType( this.GetType().Name ) ) );

			VersionNumber _vnToCompare = (VersionNumber)obj;

			if( this == _vnToCompare )
				return 0;

			int _numberOfGroupsToCheck = this._versionGroups.Length > _vnToCompare._versionGroups.Length 
				? this._versionGroups.Length : _vnToCompare._versionGroups.Length;

			for( int _groupIndex = 0; _groupIndex < _numberOfGroupsToCheck; _groupIndex++ )
			{
				int _groupCompareResult = this._versionGroups[_groupIndex].CompareTo( _vnToCompare._versionGroups[_groupIndex] );

				if( _groupCompareResult != 0 )
					return _groupCompareResult;
			}

			if( this._versionGroups.Length == _vnToCompare._versionGroups.Length )
				return 0;
			
			if( this._versionGroups.Length > _vnToCompare._versionGroups.Length )
				return DoesVersionNumberEndsWithNulls( this._versionGroups, _numberOfGroupsToCheck ) ? 0 : 1;
			else
				return DoesVersionNumberEndsWithNulls( _vnToCompare._versionGroups, _numberOfGroupsToCheck ) ? 0 : -1;
		}

		/// <summary>
		/// Method starts searching an array of integers (version number groups) from the supplied position
		/// and returns true if all numbers starting from the given position are 0s. 
		/// Returns false otherwise
		/// </summary>
		/// <param name="_versionNumberGroupsToCheck">Array of integers (version number groups) to check.</param>
		/// <param name="startingIndex">0-based index to start checking from.</param>
		/// <returns>True if supplied array ends with 0s, false otherwise.</returns>
		private static bool DoesVersionNumberEndsWithNulls( int[] _versionNumberGroupsToCheck, int startingIndex )
		{
			for( int i = startingIndex; i < _versionNumberGroupsToCheck.Length; i++ )
				if( _versionNumberGroupsToCheck[i] != 0 )
					return false;

			return true;
		}
	}
}
